/*
  Timer per ingranditore camera oscura "Timerino_3.0.8"
  2025-11-15 Alessio Bortoletto
*/

//-----------------------------------------------------------------------------------------------------
// LIBRERIE
#include <Keypad.h>
#include <avr/eeprom.h>
#include <Wire.h>
#include <LiquidCrystal_I2C.h>
#include <SoftwareSerial.h>

//-----------------------------------------------------------------------------------------------------
// 1. DEFINIZIONE COSTANTI (PIN, INDIRIZZI, COMANDI)
//-----------------------------------------------------------------------------------------------------

// --- Pin Hardware ---
const byte PIN_BUZZER = 10;
const byte PIN_RELAY = 11;
const byte PIN_MAIN_BTN = 12;
const byte PIN_OFF_SWI = 13;
const byte PIN_FUNZ_SWI = A0;
const byte PIN_PREC_SWI = A1;
const byte PIN_DISP1_TX = A2; // TX software per display 7-seg
const byte PIN_CALL_SVF = A3;

// --- Pin Keypad ---
const byte KEYPAD_ROWS = 4;
const byte KEYPAD_COLS = 4;
byte rowPins[KEYPAD_ROWS] = {6, 7, 8, 9};
byte colPins[KEYPAD_COLS] = {2, 3, 4, 5};
char keys[KEYPAD_ROWS][KEYPAD_COLS] = {
  {'1', '2', '3', 'A'},
  {'4', '5', '6', 'B'},
  {'7', '8', '9', 'C'},
  {'*', '0', '#', 'D'}
};

// --- Indirizzi I2C ---
const byte I2C_ADDR_LCD = 0x27;
const byte LCD_COLS = 16;
const byte LCD_ROWS = 2;

// --- Caratteri Speciali LCD ---
const byte CHAR_NOTE = 1;
const byte CHAR_SPINA = 2;
const byte CHAR_FRECCIA_SU = 3;
const byte CHAR_FRECCIA_GIU = 4;
uint8_t note[8] = {0x2, 0x3, 0x2, 0x2, 0xe, 0x1e, 0x1e, 0xc};
uint8_t spina[8] = {0xa, 0xa, 0x1f, 0x11, 0x11, 0xe, 0x4, 0x4};
uint8_t frecciasu[8] = {0x4, 0xe, 0x1f, 0x4, 0x4, 0x4, 0x4, 0x4};
uint8_t frecciagiu[8] = {0x4, 0x4, 0x4, 0x4, 0x4, 0x1f, 0xe, 0x4};

// --- Comandi Display 7 Segmenti ---
const byte D7SEG_CMD_RESET = 0x76;
const byte D7SEG_CMD_CURSOR = 0x77;
const byte D7SEG_CMD_BRIGHTNESS = 0x7A;
const byte D7SEG_DOT_NONE = 0x00;
const byte D7SEG_DOT_DECIMAL = 0x04; // Punto decimale alla seconda cifra

// --- Toni Buzzer ---
const int TONE_UP = 600;
const int TONE_DOWN = 300;

// --- Mappa EEPROM ---
const int EEPROM_ADDR_BRIGHTNESS = 1;   // 2 byte (word)
const int EEPROM_ADDR_MCD_START = 3;    // 60 byte (3 ch * 10 step * 2 byte)
const int EEPROM_ADDR_ATTREL_START = 83;  // 3 byte (per ch 1, 2, 3)
const int EEPROM_ADDR_PREORIL = 88;   // 1 byte
const int EEPROM_ADDR_SVFAT_START = 89;   // 60 byte (3 ch * 10 step * 2 byte)
const int EEPROM_ADDR_HOUR_COUNTER = 200; // 4 byte (dword)
const int EEPROM_CHANNEL_SIZE_MCD = 20;   // 10 step * 2 byte
const int EEPROM_CHANNEL_SIZE_SVFAT = 20;   // 10 step * 2 byte

// --- Modalità Timer (Costanti) ---
const byte FOCUS     =  1; // 01 Impostazioni e messa a fuoco
const byte MCDOWN    =  2; // 02 Conto alla rovescia multiplo
const byte PSCAL     =  3; // 03 Provino a scalare lineare
const byte CUP       =  4; // 04 Conto in avanti
const byte CDOWN     =  5; // 05 Conto alla rovescia
const byte CDOWN2    =  6; // 06 Conto alla rovescia con primo giro a vuoto
const byte PSCAF     =  7; // 07 Provino a scalare f-stop
const byte CFSTOP    =  8; // 08 Calcolatore f-stop
const byte CFSTOP2   =  9; // 09 Calcolatore f-stop con primo giro a vuoto
const byte MBFSTP    = 10; // 10 Scherma-Brucia f-stop
const byte SVFAT     = 11; // 11 Sviluppo stampa fattoriale

//-----------------------------------------------------------------------------------------------------
// 2. DEFINIZIONE STATI E VARIABILI GLOBALI
//-----------------------------------------------------------------------------------------------------

// --- Stati Programma ---
enum ProgramState {
  STATE_IDLE, STATE_COUNTDOWN, STATE_MCDOWN, STATE_STOPWATCH, STATE_TEST_STRIP,
  STATE_MBFSTP_BURN, STATE_MBFSTP_EXPOSURE, STATE_SVFAT_COUNTUP, STATE_SVFAT_FACTORIAL_DOWN, STATE_SVFAT_COUNTDOWN,
  STATE_WAITING_FOR_SECOND_PRESS, STATE_HOUR_COUNTER_DISPLAY
};
ProgramState currentState = STATE_IDLE;

// --- Variabili Globali ---
boolean isFocusRelayOn = false;
boolean beepSequenceActive = false; byte beepStep = 0; byte beepSequenceLength = 0;
unsigned long lastBeepActionTime = 0; int currentBeepTone = 0; int currentBeepDuration = 0;
int tsvfat_time[10][4]; byte isvfat[4] = {0, 0, 0, 0};
boolean mute = false; boolean attrel[4]; boolean provcomp = true;
byte timer_mode = 0; byte ch = 1; int brightness = 0; int preoril;
byte ncifra = 0; int time = 0; int time_succ; int appo_time; int time_burn; int initial_time_burn = 0;
int time_countdown[4]; int time_dds[4]; int time_fsttest[4]; int time_fstdown[4];
int time_fstdown2[4]; int time_brn[4]; int time_test[4]; int time_svfatt[4];
int precis = 1; int iprec = 0; int last_timer_mod = -1; int last_iprec = -1;
byte sca = 0; int svfatInputIndex = 0;
int tmcd_time[10][4]; byte imcd[4] = {0, 0, 0, 0};
long last_time = 0; float mult = 1.0; char _buffer[17];
bool isDodgeOperation = false;
bool isTimerPaused = false;

// --- Flag per salvataggio sicuro EEPROM Contaore ---
bool hour_counter_is_dirty = false;

// --- Variabili Scorrimento Testo ---
char sequence_string[100];
bool is_scrolling_active = false;
int scroll_index = 0;
unsigned long last_scroll_time = 0;
const int scroll_speed_ms = 400;
const int scroll_pause_duration_ms = 1000;
enum ScrollState { SCROLLING, PAUSED, STATIC_DISPLAY };
ScrollState scroll_state = SCROLLING;
unsigned long state_timer = 0;
const int scroll_static_duration_ms = 3000;

// --- Variabili Contaore Lampada ---
unsigned long total_relay_seconds = 0;
unsigned long relay_on_time = 0;

//-----------------------------------------------------------------------------------------------------
// 3. OGGETTI
//-----------------------------------------------------------------------------------------------------
Keypad keypad = Keypad( makeKeymap(keys), rowPins, colPins, KEYPAD_ROWS, KEYPAD_COLS);
SoftwareSerial d7seg(255, PIN_DISP1_TX);
LiquidCrystal_I2C lcd(I2C_ADDR_LCD, LCD_COLS, LCD_ROWS);

//-----------------------------------------------------------------------------------------------------
// 4. PROTOTIPI DI FUNZIONE
//-----------------------------------------------------------------------------------------------------
void say_time(int ar_time); void init_timermode(); void say_timermode(); void set_precis();
void say_ch(); void say_clr(); void say_reset(); void say_clearlcd();
void say_sound(); void say_rel(); void say_preoril(); void beep(int this_tone, int duration);
void metronome(int ar_time); void metronomeSVFAT(int ar_time); void load_eeprom();
void write_eeprom(); void say_tipro(); 
void readKeypad(bool &state_changed); // <-- MODIFICA PROTOTIPO
void readSwitches();
void say_precis(); void say_ok(); void say_err(); void say_reg(); void say_rst();
void startBeepSequence(byte numBeeps, int tone, int duration);
void say_version();
void handleIdleState(bool buttonPressed);
void handleCountdownState(bool buttonPressed);
void handleMcdownState(bool buttonPressed);
void handleSvfCountUpState(bool buttonPressed);
void handleSvfFactorialDownState(bool buttonPressed);
void handleSvfCountdownState(bool buttonPressed);
void handleStopwatchState(bool buttonPressed);
void handleTestStripState(bool buttonPressed);
void handleWaitingForSecondPressState(bool buttonPressed);
void handleMbFstpBurnState(bool buttonPressed);
void handleMbFstpExposureState(bool buttonPressed);
void handleHourCounterDisplayState();
void handleBeeps();
void clearInputTime();
void say_mute();
void reset_mcdown_channel(byte channel);
void reset_svfat_channel(byte channel);
void reset_hour_counter();
void update_icons();
void build_sequence_string();
void handle_sequence_display();
void clear_second_line_content();
void turn_relay_on();
void turn_relay_off();
void assign_time_to_channel(byte target_channel);

//-----------------------------------------------------------------------------------------------------
// 5. SETUP
//-----------------------------------------------------------------------------------------------------
void setup() {
  Serial.begin(9600);
  pinMode(PIN_FUNZ_SWI, INPUT);
  pinMode(PIN_PREC_SWI, INPUT);
  pinMode(PIN_DISP1_TX, OUTPUT);
  pinMode(PIN_BUZZER, OUTPUT);
  pinMode(PIN_RELAY, OUTPUT);
  digitalWrite(PIN_RELAY, HIGH);
  pinMode(PIN_MAIN_BTN, INPUT);
  pinMode(PIN_OFF_SWI, INPUT_PULLUP);
  pinMode(PIN_CALL_SVF, INPUT_PULLUP);
  
  keypad.setDebounceTime(50);
  load_eeprom();
  
  lcd.init();
  lcd.backlight();
  lcd.createChar(CHAR_NOTE, note);
  lcd.createChar(CHAR_SPINA, spina);
  lcd.createChar(CHAR_FRECCIA_SU, frecciasu);
  lcd.createChar(CHAR_FRECCIA_GIU, frecciagiu);
  
  d7seg.begin(9600);
  d7seg.write(D7SEG_CMD_RESET);
  d7seg.write(D7SEG_CMD_BRIGHTNESS);
  d7seg.write((byte) brightness);
  
  say_version();
  say_reset();
  say_clearlcd();
  currentState = STATE_IDLE;
  init_timermode();
}

//-----------------------------------------------------------------------------------------------------
// 6. LOOP PRINCIPALE (MODIFICATO)
//-----------------------------------------------------------------------------------------------------
void loop() {
  handle_sequence_display();
  handleBeeps();
  
  static unsigned long lastButtonCheck = 0;
  static bool lastButtonState = false;
  bool buttonPressed = false;
  
  if (millis() - lastButtonCheck > 50) {
    bool currentButtonState = (digitalRead(PIN_MAIN_BTN) == HIGH);

    if (preoril == LOW) {
      if (currentButtonState && !lastButtonState) {
        buttonPressed = true;
      }
    } else {
      if (!currentButtonState && lastButtonState) {
        buttonPressed = true;
      }
    }
    
    lastButtonState = currentButtonState;
    lastButtonCheck = millis();
  }

  // *** MODIFICA: Aggiunto flag 'state_was_changed_by_keypad' ***
  bool state_was_changed_by_keypad = false;

  switch (currentState) {
    case STATE_IDLE: 
      readSwitches(); // readSwitches va prima
      readKeypad(state_was_changed_by_keypad);
      if (state_was_changed_by_keypad) break; // Interrompe il loop se 'D' o 'A'/'B'/'C' sono stati premuti
      handleIdleState(buttonPressed); 
      break;
    case STATE_COUNTDOWN: 
      readKeypad(state_was_changed_by_keypad);
      if (state_was_changed_by_keypad) break;
      handleCountdownState(buttonPressed); 
      break; 
    case STATE_MCDOWN: 
      readKeypad(state_was_changed_by_keypad);
      if (state_was_changed_by_keypad) break;
      handleMcdownState(buttonPressed); 
      break;
    case STATE_STOPWATCH: 
      handleStopwatchState(buttonPressed); 
      break;
    case STATE_TEST_STRIP: 
      readKeypad(state_was_changed_by_keypad);
      if (state_was_changed_by_keypad) break;
      handleTestStripState(buttonPressed); 
      break;
    case STATE_MBFSTP_BURN: 
      readKeypad(state_was_changed_by_keypad);
      if (state_was_changed_by_keypad) break;
      handleMbFstpBurnState(buttonPressed); 
      break; 
    case STATE_MBFSTP_EXPOSURE: 
      readKeypad(state_was_changed_by_keypad);
      if (state_was_changed_by_keypad) break;
      handleMbFstpExposureState(buttonPressed); 
      break; 
    case STATE_SVFAT_COUNTUP: 
      handleSvfCountUpState(buttonPressed); 
      break;
    case STATE_SVFAT_FACTORIAL_DOWN: 
      readKeypad(state_was_changed_by_keypad);
      if (state_was_changed_by_keypad) break;
      handleSvfFactorialDownState(buttonPressed); 
      break;
    case STATE_SVFAT_COUNTDOWN: 
      readKeypad(state_was_changed_by_keypad);
      if (state_was_changed_by_keypad) break;
      handleSvfCountdownState(buttonPressed); 
      break;
    case STATE_WAITING_FOR_SECOND_PRESS: 
      readSwitches(); 
      readKeypad(state_was_changed_by_keypad);
      if (state_was_changed_by_keypad) break;
      handleWaitingForSecondPressState(buttonPressed); 
      break;
    case STATE_HOUR_COUNTER_DISPLAY: 
      readKeypad(state_was_changed_by_keypad);
      if (state_was_changed_by_keypad) break;
      handleHourCounterDisplayState(); 
      break;
  }
}

//-----------------------------------------------------------------------------------------------------
// 7. FUNZIONI DI GESTIONE STATI
//-----------------------------------------------------------------------------------------------------

// *** MODIFICA: Rimossi readSwitches() e readKeypad() ***
void handleIdleState(bool buttonPressed) {
  // readSwitches() e readKeypad() sono ora chiamati nel loop()
  if (buttonPressed) {
    switch (timer_mode) {
      case FOCUS:
        if (attrel[ch]) {
          if (isFocusRelayOn) {
            turn_relay_off(); isFocusRelayOn = false;
          } else {
            turn_relay_on(); isFocusRelayOn = true;
          }
        }
        break;
      case CUP:
        last_time = millis(); if (attrel[ch]) { turn_relay_on(); }
        currentState = STATE_STOPWATCH; 
        break;
      case MCDOWN:
        if (imcd[ch] > 0 && tmcd_time[0][ch] > 0) {
          time = tmcd_time[0][ch];
          sca = 0;
          last_time = millis();
          if (attrel[ch]) {
            turn_relay_on();
          }
          isTimerPaused = false;
          currentState = STATE_MCDOWN;
        }
        break;
      case CDOWN: case CFSTOP:
        if (time > 0) {
          appo_time = time; last_time = millis();
          isTimerPaused = false;
          if (attrel[ch]) { turn_relay_on(); }
          currentState = STATE_COUNTDOWN;
        }
        break;
      case CDOWN2: case CFSTOP2:
        if (time > 0) {
          appo_time = time;
          if (attrel[ch]) { turn_relay_on(); }
          currentState = STATE_WAITING_FOR_SECOND_PRESS;
        }
        break;
      case PSCAL: case PSCAF:
        appo_time = time; 
        if (time < 10) { 
          time = 10; 
          appo_time = 10;
        }
        time_succ = appo_time; 
        time = 0; 
        last_time = millis();
        if (attrel[ch]) { turn_relay_on(); }
        isTimerPaused = false; 
        currentState = STATE_TEST_STRIP;
        break;
      case MBFSTP:
        if (time > 0) {
          appo_time = time;
          if (time_burn > 0) {
            if (attrel[ch]) { turn_relay_on(); }
            last_time = millis(); currentState = STATE_MBFSTP_BURN;
          } else {
            if (attrel[ch]) { turn_relay_on(); }
            isTimerPaused = false;
            last_time = millis(); 
            currentState = STATE_MBFSTP_EXPOSURE;
          }
        }
        break;
      case SVFAT:
        if (isvfat[ch] > 1) {
          time = 0; last_time = millis(); isTimerPaused = false;
          currentState = STATE_SVFAT_COUNTUP;
        }
        break;
    }
  }
}

void handleCountdownState(bool buttonPressed) {
  if (currentState != STATE_COUNTDOWN && currentState != STATE_MBFSTP_EXPOSURE) return;

  if (buttonPressed) {
    isTimerPaused = !isTimerPaused;
    if (isTimerPaused) {
      if (attrel[ch]) { turn_relay_off(); }
    } else {
      if (attrel[ch]) { turn_relay_on(); }
      last_time = millis();
    }
    return;
  }

  if (!isTimerPaused) {
    if (millis() - last_time >= 100) {
      last_time = millis();
      time--;
      metronome(time);
      say_time(time);
      if (time <= 0) {
        if (attrel[ch]) { turn_relay_off(); }
        startBeepSequence(3, TONE_DOWN, 150);
        time = appo_time;
        say_time(time);
        currentState = STATE_IDLE;
      }
    }
  }
}

void handleMcdownState(bool buttonPressed) {
  if (currentState != STATE_MCDOWN) return;

  if (buttonPressed) {
    isTimerPaused = !isTimerPaused;
    if (isTimerPaused) {
      if (attrel[ch]) { turn_relay_off(); }
    } else {
      if (attrel[ch]) { turn_relay_on(); }
      last_time = millis();
    }
    return;
  }

  if (!isTimerPaused) {
    if (millis() - last_time >= 100) {
      last_time = millis();
      time--;
      say_time(time);
      if (time <= 0) {
        startBeepSequence(3, TONE_UP, 75);
        sca++;
        if (sca < imcd[ch]) {
          time = tmcd_time[sca][ch];
          say_time(time);
        } else {
          if(attrel[ch]) { turn_relay_off(); }
          time = tmcd_time[0][ch];
          say_time(time);
          currentState = STATE_IDLE;
        }
      }
    }
  }
}

void handleSvfCountUpState(bool buttonPressed) {
  if (buttonPressed) {
    int calculated_time = time * tsvfat_time[0][ch] - time;
    if (calculated_time < 0) calculated_time = 0;
    time = calculated_time; last_time = millis(); sca = 1;
    currentState = STATE_SVFAT_FACTORIAL_DOWN; return;
  }
  if (millis() - last_time >= 100) {
    last_time = millis(); time++; metronome(time); say_time(time);
  }
}

void handleSvfFactorialDownState(bool buttonPressed) {
  if (currentState != STATE_SVFAT_FACTORIAL_DOWN) return;
  if (buttonPressed) { isTimerPaused = !isTimerPaused; if (!isTimerPaused) { last_time = millis(); } }
  if (!isTimerPaused) {
    if (millis() - last_time >= 100) {
      last_time = millis(); time--; metronomeSVFAT(time); say_time(time);
      if (time <= 0) {
        startBeepSequence(3, TONE_UP, 75);
        if (sca < isvfat[ch]) {
          currentState = STATE_SVFAT_COUNTDOWN; time = tsvfat_time[sca][ch]; say_time(time);
        } else {
          currentState = STATE_IDLE; time = tsvfat_time[0][ch]; say_time(time);
        }
      }
    }
  }
}

void handleSvfCountdownState(bool buttonPressed) {
  if (currentState != STATE_SVFAT_COUNTDOWN) return;
  if (buttonPressed) { isTimerPaused = !isTimerPaused; if (!isTimerPaused) { last_time = millis(); } }
  if (!isTimerPaused) {
    if (millis() - last_time >= 100) {
      last_time = millis(); time--; metronomeSVFAT(time); say_time(time);
      if (time <= 0) {
        startBeepSequence(3, TONE_UP, 75); sca++;
        if (sca < isvfat[ch]) {
          time = tsvfat_time[sca][ch]; say_time(time);
        } else {
          currentState = STATE_IDLE; time = tsvfat_time[0][ch]; say_time(time);
        }
      }
    }
  }
}

void handleStopwatchState(bool buttonPressed) {
  if (buttonPressed) {
    if (attrel[ch]) { turn_relay_off(); }
    currentState = STATE_IDLE; return;
  }
  if (millis() - last_time >= 100) {
    last_time = millis(); time++; metronome(time); say_time(time);
  }
}


void handleTestStripState(bool buttonPressed) {
  if (buttonPressed) {
    isTimerPaused = !isTimerPaused;
    if (isTimerPaused) {
      if (attrel[ch]) { turn_relay_off(); }
    } else {
      if (attrel[ch]) { turn_relay_on(); }
      last_time = millis();
    }
    return;
  }

  if (!isTimerPaused) {
    if (millis() - last_time >= 100) {
      last_time = millis(); 
      time++; 
      say_time(time);
      
      if (time >= time_succ) {
        beep(TONE_UP, 150);
        if (timer_mode == PSCAF) {
          mult = pow(2.0, (1.0 / precis)); time_succ = round((float)time_succ * mult);
        } else {
          time_succ = time_succ + (precis * 10);
        }
        if (provcomp) {
          if (attrel[ch]) { turn_relay_off(); }
          say_time(time_succ); 
          currentState = STATE_WAITING_FOR_SECOND_PRESS;
        }
      }
      if (time_succ > time && time_succ - time <= 3) { beep(TONE_UP, 50); }
    }
  }
}


void handleWaitingForSecondPressState(bool buttonPressed) {
  if (buttonPressed) {
    if (attrel[ch]) {
      turn_relay_on();
    }
    last_time = millis();

    if ((timer_mode == PSCAL || timer_mode == PSCAF) && provcomp) {
      time = 0;
      isTimerPaused = false; 
      currentState = STATE_TEST_STRIP;
    } else if (timer_mode == MBFSTP && isDodgeOperation) {
      isTimerPaused = false;
      currentState = STATE_MBFSTP_EXPOSURE;
    } else {
      isTimerPaused = false;
      currentState = STATE_COUNTDOWN;
    }
  }
}

void handleMbFstpBurnState(bool buttonPressed) {
  if (buttonPressed) {
    isTimerPaused = !isTimerPaused;
    if (isTimerPaused) {
      if (attrel[ch]) { turn_relay_off(); }
    } else {
      if (attrel[ch]) { turn_relay_on(); }
      last_time = millis();
    }
    return;
  }
  
  if (!isTimerPaused) {
    if (millis() - last_time >= 100) {
      last_time = millis();
      time_burn--;
      metronome(time_burn);
      say_time(time_burn);
      if (time_burn <= 0) {
        if (attrel[ch]) { turn_relay_off(); }

        if (isDodgeOperation) {
          time = appo_time - initial_time_burn;
          appo_time = time;
        } else {
          time = appo_time;
        }
        say_time(time);
        currentState = STATE_WAITING_FOR_SECOND_PRESS;
      }
    }
  }
}

void handleMbFstpExposureState(bool buttonPressed) {
  handleCountdownState(buttonPressed);
}

// *** MODIFICA: Rimossa chiamata a readKeypad() ***
void handleHourCounterDisplayState() {
  // readKeypad() è ora chiamato nel loop()
  if (millis() - last_time > 4000) {
    currentState = STATE_IDLE;
    init_timermode();
  }
}

//-----------------------------------------------------------------------------------------------------
// 8. FUNZIONI DI SUPPORTO
//-----------------------------------------------------------------------------------------------------

void assign_time_to_channel(byte target_channel) {
  if (ncifra > 0) { 
    switch (timer_mode) {
      case CDOWN:   time_countdown[target_channel] = time; break;
      case CDOWN2:  time_dds[target_channel] = time; break;
      case PSCAL:   time_test[target_channel] = time; break;
      case PSCAF:   time_fsttest[target_channel] = time; break;
      case CFSTOP:  time_fstdown[target_channel] = time; break;
      case CFSTOP2: time_fstdown2[target_channel] = time; break;
      case MBFSTP:  time_brn[target_channel] = time; break;
    }
  }
}


void turn_relay_on() {
  if (digitalRead(PIN_RELAY) == HIGH) {
    digitalWrite(PIN_RELAY, LOW);
    relay_on_time = millis();
  }
}

void turn_relay_off() {
  if (digitalRead(PIN_RELAY) == LOW) {
    unsigned long duration_ms = millis() - relay_on_time;
    total_relay_seconds += duration_ms / 1000;
    digitalWrite(PIN_RELAY, HIGH);
    
    hour_counter_is_dirty = true; 
  }
}

void clear_second_line_content() {
  lcd.setCursor(0, 1);
  lcd.print(F("                "));
}

void build_sequence_string() {
  sequence_string[0] = '\0';
  char temp_buffer[8];
  byte num_steps = 0;
  int start_index = 0;
  int (*time_array)[4] = NULL;

  if (timer_mode == MCDOWN) {
    num_steps = imcd[ch]; time_array = tmcd_time; start_index = 0;
  } else if (timer_mode == SVFAT) {
    num_steps = isvfat[ch]; time_array = tsvfat_time; start_index = 1;
  }

  if (num_steps > start_index) {
    for (int i = start_index; i < num_steps; i++) {
      int seconds = time_array[i][ch] / 10;
      sprintf(temp_buffer, "%d", seconds);
      strcat(sequence_string, temp_buffer);
      if (i < num_steps - 1) { strcat(sequence_string, "-"); }
    }
  }
}

void handle_sequence_display() {
  if (!is_scrolling_active) return;
  int len = strlen(sequence_string);

  switch (scroll_state) {
    case SCROLLING:
      if (millis() - last_scroll_time > scroll_speed_ms) {
        last_scroll_time = millis();
        char display_buffer[13];
        memset(display_buffer, ' ', 12); display_buffer[12] = '\0';
        int chars_to_copy = min(len - scroll_index, 12);
        if (chars_to_copy > 0) { strncpy(display_buffer, sequence_string + scroll_index, chars_to_copy); }
        lcd.setCursor(0, 1); lcd.print(display_buffer);
        scroll_index++;
        if (scroll_index > len) {
          scroll_state = PAUSED; state_timer = millis();
          lcd.setCursor(0, 1); lcd.print(F("            "));
        }
      }
      break;
    case PAUSED:
      if (millis() - state_timer > scroll_pause_duration_ms) {
        scroll_state = STATIC_DISPLAY; state_timer = millis();
        char static_buffer[13];
        strncpy(static_buffer, sequence_string, 12); static_buffer[12] = '\0';
        for (int i = strlen(static_buffer); i < 12; i++) { static_buffer[i] = ' '; }
        lcd.setCursor(0, 1); lcd.print(static_buffer);
      }
      break;
    case STATIC_DISPLAY:
      if (millis() - state_timer > scroll_static_duration_ms) {
        scroll_state = SCROLLING; scroll_index = 0; last_scroll_time = millis();
      }
      break;
  }
}


void clearInputTime() {
  time = 0;
  ncifra = 1; 
  assign_time_to_channel(ch);
  
  ncifra = 0; 
  init_timermode(); 
  currentState = STATE_IDLE;
}

void reset_mcdown_channel(byte channel) {
  imcd[channel] = 0;
  for (int i = 0; i < 10; i++) { tmcd_time[i][channel] = 0; }
  time = 0; ncifra = 0; say_time(time); write_eeprom(); init_timermode();
}

void reset_svfat_channel(byte channel) {
  isvfat[channel] = 0;
  for (int i = 0; i < 10; i++) { tsvfat_time[i][channel] = 0; }
  time = 0; ncifra = 0; say_time(time); write_eeprom(); init_timermode();
}

void reset_hour_counter() {
  total_relay_seconds = 0;
  eeprom_write_dword((uint32_t*)EEPROM_ADDR_HOUR_COUNTER, total_relay_seconds);
  lcd.clear();
  lcd.setCursor(0, 0);
  lcd.print(F("Contaore"));
  lcd.setCursor(0, 1);
  lcd.print(F("Azzerato!"));
  delay(1500);
}


// *** MODIFICA: Aggiunto 'state_changed' e 'return;' ***
void readKeypad(bool &state_changed) {
  char key = keypad.getKey();
  if (!key) return;

  // Resetta lo stato se si preme A/B/C durante un'operazione
  if (currentState != STATE_IDLE && (key == 'A' || key == 'B' || key == 'C')) {
    if (attrel[ch]) {
      turn_relay_off();
    }
    isTimerPaused = false;
    currentState = STATE_IDLE;
  }

  switch (key) {
    case 'A': 
      if (timer_mode == FOCUS) {
         ch = 1;
      } else {
         assign_time_to_channel(1);
         ch = 1;
      }
      beep(TONE_DOWN, 200);
      init_timermode();
      state_changed = true; // <-- CORREZIONE
      return; // <-- CORREZIONE
      
    case 'B': 
      if (timer_mode == FOCUS) {
         ch = 2;
      } else {
         assign_time_to_channel(2);
         ch = 2;
      }
      beep(TONE_DOWN, 200);
      init_timermode();
      state_changed = true; // <-- CORREZIONE
      return; // <-- CORREZIONE
      
    case 'C': 
      if (timer_mode == FOCUS) {
         ch = 3;
      } else {
         assign_time_to_channel(3);
         ch = 3;
      }
      beep(TONE_DOWN, 200);
      init_timermode();
      state_changed = true; // <-- CORREZIONE
      return; // <-- CORREZIONE
    
    case 'D':
      if (timer_mode == FOCUS) {
        say_clr();
        delay(250);
        reset_hour_counter();
        init_timermode();
        currentState = STATE_IDLE;
        state_changed = true; // <-- CORREZIONE
        return; // <-- CORREZIONE
      }
      // Logica "Rst"
      else if ((timer_mode == MCDOWN || timer_mode == SVFAT || 
                timer_mode == MBFSTP) 
                && currentState != STATE_IDLE) 
      {
        say_rst();
        delay(250);
        isTimerPaused = false;
        currentState = STATE_IDLE;
        init_timermode();
        state_changed = true; // <-- CORREZIONE
        return; // <-- CORREZIONE
      } 
      // Logica "Clr"
      else {
        say_clr();
        delay(250);
        
        if (currentState != STATE_IDLE) {
          isTimerPaused = false;
          currentState = STATE_IDLE;
        }

        if (timer_mode == MCDOWN) {
          reset_mcdown_channel(ch);
        } else if (timer_mode == SVFAT) {
          reset_svfat_channel(ch);
        } else {
          clearInputTime(); // Chiama init_timermode() internamente
        }
        state_changed = true; // <-- CORREZIONE
        return; // <-- CORREZIONE
      }
      
    case '*':
      if (timer_mode == FOCUS) {
        if (brightness > 0) {
          brightness -= 10; 
          d7seg.write(D7SEG_CMD_BRIGHTNESS); 
          d7seg.write((byte)brightness);
          eeprom_write_word((uint16_t *)EEPROM_ADDR_BRIGHTNESS, brightness);
        }
      } else if (timer_mode == PSCAL || timer_mode == PSCAF) {
        provcomp = false; say_tipro();
      } else if (timer_mode == MCDOWN) {
        if (time != 0) {
          if (imcd[ch] < 10) {
            tmcd_time[imcd[ch]][ch] = time; say_ok(); delay(500);
            time = 0; ncifra = 0; say_time(time); imcd[ch]++;
          } else { say_err(); delay(500); }
        }
      } else if (timer_mode == SVFAT) {
        if (time != 0) {
          if (isvfat[ch] < 10) {
            tsvfat_time[isvfat[ch]][ch] = time; say_ok(); delay(500);
            time = 0; ncifra = 0; say_time(time); isvfat[ch]++;
          } else { say_err(); delay(500); }
        }
      } else if (timer_mode == CDOWN || timer_mode == CDOWN2) {
        int step = precis * 10; time -= step; if (time < 0) time = 0; say_time(time);
        ncifra = 0;
      } else if (timer_mode == CFSTOP || timer_mode == CFSTOP2) {
        float mult = pow(2.0, (1.0 / precis)); time = round((float)time / mult);
        if (time < 0) time = 0; say_time(time);
        ncifra = 0;
      } else if (timer_mode == MBFSTP) {
        isDodgeOperation = true; mult = 1.0 / precis;
        time_burn = round((float)time * mult); initial_time_burn = time_burn; say_time(time_burn);
        ncifra = 0;
      }
      break;
    case '#':
      if (timer_mode == FOCUS) {
        if (brightness < 100) {
          brightness += 10; 
          d7seg.write(D7SEG_CMD_BRIGHTNESS); 
          d7seg.write((byte)brightness);
          eeprom_write_word((uint16_t *)EEPROM_ADDR_BRIGHTNESS, brightness);
        }
      } else if (timer_mode == PSCAL || timer_mode == PSCAF) {
        provcomp = true; say_tipro();
      } else if (timer_mode == MCDOWN) {
        if (imcd[ch] != 0 || time != 0) {
          if (time != 0) {
            if (imcd[ch] < 10) { tmcd_time[imcd[ch]][ch] = time; imcd[ch]++; }
          }
          write_eeprom(); say_reg(); delay(500);
          time = tmcd_time[0][ch]; say_time(time); say_mcdown();
        }
      } else if (timer_mode == SVFAT) {
        if (isvfat[ch] != 0 || time != 0) {
          if (time != 0) {
            if (isvfat[ch] < 10) { tsvfat_time[isvfat[ch]][ch] = time; isvfat[ch]++; }
          }
          write_eeprom(); say_reg(); delay(500);
          time = tsvfat_time[0][ch]; say_time(time); say_svfatt();
        }
      } else if (timer_mode == CDOWN || timer_mode == CDOWN2) {
        int step = precis * 10; time += step; if (time > 9999) time = 9999; say_time(time);
        ncifra = 0;
      } else if (timer_mode == CFSTOP || timer_mode == CFSTOP2) {
        float mult = pow(2.0, (1.0 / precis)); time = round((float)time * mult);
        if (time > 9999) time = 9999; say_time(time);
        ncifra = 0;
      } else if (timer_mode == MBFSTP) {
        isDodgeOperation = false; mult = 1.0 / precis;
        time_burn = round((float)time * mult); initial_time_burn = time_burn; say_time(time_burn);
        ncifra = 0;
      }
      break;
    default:
      if (isdigit(key)) {
        if (timer_mode == FOCUS) {
          switch (key) {
            case '0':
              mute = !mute; beep(TONE_DOWN, 200); update_icons(); break;
            case '1':
              attrel[ch] = !attrel[ch]; 
              eeprom_write_byte((uint8_t *)(EEPROM_ADDR_ATTREL_START + ch), attrel[ch]);
              beep(TONE_DOWN, 200); 
              update_icons(); 
              break;
            case '2':
              preoril = (preoril == LOW) ? HIGH : LOW; 
              eeprom_write_byte((uint8_t *)EEPROM_ADDR_PREORIL, (preoril == HIGH));
              beep(TONE_DOWN, 200); 
              update_icons(); 
              break;
            case '3':
              char buffer[17];
              unsigned long hours = total_relay_seconds / 3600;
              unsigned int minutes = (total_relay_seconds % 3600) / 60;
              lcd.clear();
              lcd.setCursor(0, 0);
              lcd.print(F("Uso Lampada:"));
              sprintf(buffer, "%lu h %u min", hours, minutes);
              lcd.setCursor(0, 1);
              lcd.print(buffer);
              currentState = STATE_HOUR_COUNTER_DISPLAY;
              last_time = millis();
              state_changed = true; // <-- CORREZIONE
              break;
          }
        } else if (timer_mode != CUP) {
          if (ncifra == 0) time = 0; 
          if (time < 1000) { time = time * 10 + (key - '0'); }
          ncifra++;
          say_time(time);
        }
      }
      break;
  }
}

void readSwitches() {
  int swistate = digitalRead(PIN_OFF_SWI);
  int svfstate = digitalRead(PIN_CALL_SVF);
  int new_timer_mode;

  if (swistate == LOW && svfstate == LOW) {
    new_timer_mode = ((analogRead(PIN_FUNZ_SWI) + 47) * 11 / 1023) + 1;
  } else {
    new_timer_mode = (svfstate == LOW) ? SVFAT : FOCUS;
  }

  if (new_timer_mode != timer_mode) {
    
    if (digitalRead(PIN_RELAY) == LOW) {
       turn_relay_off();
    }
    if (timer_mode == FOCUS && isFocusRelayOn) { 
      isFocusRelayOn = false; 
    }
    
    if (hour_counter_is_dirty) {
      eeprom_write_dword((uint32_t*)EEPROM_ADDR_HOUR_COUNTER, total_relay_seconds);
      hour_counter_is_dirty = false;
    }
    
    timer_mode = new_timer_mode; 
    init_timermode();
    currentState = STATE_IDLE;
  }

  int p_val_raw = (analogRead(PIN_PREC_SWI) + 47) * 11 / 1023;
  int new_iprec = (timer_mode == PSCAF || timer_mode == CFSTOP || timer_mode == CFSTOP2 || timer_mode == MBFSTP) ? p_val_raw + 11 : p_val_raw;

  if (new_iprec != iprec) { iprec = new_iprec; set_precis(); }
}

void say_time(int ar_time) {
  d7seg.write(D7SEG_CMD_CURSOR);
  bool isRunning = (currentState != STATE_IDLE && currentState != STATE_WAITING_FOR_SECOND_PRESS);
  if (timer_mode == SVFAT) {
    d7seg.write((uint8_t)((!isRunning && (isvfat[ch] == 0 || ar_time == tsvfat_time[0][ch])) ? D7SEG_DOT_NONE : D7SEG_DOT_DECIMAL));
  } else { 
    d7seg.write((uint8_t)D7SEG_DOT_DECIMAL); 
  }
  sprintf(_buffer, "%04i", ar_time); 
  d7seg.write(_buffer);
}


void init_timermode() {
  clear_second_line_content();
  ncifra = 0; 
  is_scrolling_active = false;
  
  switch (timer_mode) {
    case CDOWN:   time = time_countdown[ch]; break;
    case CDOWN2:  time = time_dds[ch]; break;
    case PSCAL:   time = time_test[ch]; break;
    case PSCAF:   time = time_fsttest[ch]; break;
    case CFSTOP:  time = time_fstdown[ch]; break;
    case CFSTOP2: time = time_fstdown2[ch]; break;
    case MBFSTP:  time = time_brn[ch]; break;
    case MCDOWN:  time = tmcd_time[0][ch]; sca = 0; break;
    case FOCUS:   time = 0; break;
    case SVFAT:   time = tsvfat_time[0][ch]; sca = 0; break;
    case CUP:     time = 0; break;
  }

  say_timermode(); set_precis(); say_time(time); update_icons(); say_ch();
}

void say_clearlcd() { lcd.clear(); }
void say_ch() { lcd.setCursor(15, 1); lcd.print((char)('A' + ch - 1)); }

void say_precis() {
  if (timer_mode == PSCAL || timer_mode == CDOWN || timer_mode == CDOWN2 ||
      timer_mode == PSCAF || timer_mode == CFSTOP || timer_mode == CFSTOP2 ||
      timer_mode == MBFSTP) {
    switch (iprec) {
      case 0: lcd.setCursor(0, 1); lcd.print(F("Pr:20s  ")); break;
      case 1: lcd.setCursor(0, 1); lcd.print(F("Pr:15s  ")); break;
      case 2: lcd.setCursor(0, 1); lcd.print(F("Pr:10s  ")); break;
      case 3: lcd.setCursor(0, 1); lcd.print(F("Pr: 7s  ")); break;
      case 4: lcd.setCursor(0, 1); lcd.print(F("Pr: 5s  ")); break;
      case 5: lcd.setCursor(0, 1); lcd.print(F("Pr: 3s  ")); break;
      case 6: lcd.setCursor(0, 1); lcd.print(F("Pr: 2s  ")); break;
      case 7: case 8: case 9: case 10: lcd.setCursor(0, 1); lcd.print(F("Pr: 1s  ")); break;
      case 11: lcd.setCursor(0, 1); lcd.print(F("Pr:1/1  ")); break;
      case 12: lcd.setCursor(0, 1); lcd.print(F("Pr:1/2  ")); break;
      case 13: lcd.setCursor(0, 1); lcd.print(F("Pr:1/3  ")); break;
      case 14: lcd.setCursor(0, 1); lcd.print(F("Pr:1/4  ")); break;
      case 15: lcd.setCursor(0, 1); lcd.print(F("Pr:1/6  ")); break;
      case 16: lcd.setCursor(0, 1); lcd.print(F("Pr:1/8  ")); break;
      case 17: lcd.setCursor(0, 1); lcd.print(F("Pr:1/12 ")); break;
      case 18: lcd.setCursor(0, 1); lcd.print(F("Pr:1/16 ")); break;
      case 19: lcd.setCursor(0, 1); lcd.print(F("Pr:1/24 ")); break;
      case 20: lcd.setCursor(0, 1); lcd.print(F("Pr:1/32 ")); break;
      case 21: lcd.setCursor(0, 1); lcd.print(F("Pr:1/64 ")); break;
    }
  }
}

void say_free() {
  lcd.setCursor(0, 0); lcd.print(F("Focus      SetUp"));
}
void say_mcdown() {
  lcd.setCursor(0, 0); lcd.print(F("Multi Count Down")); say_time(tmcd_time[0][ch]);
  build_sequence_string(); int len = strlen(sequence_string);
  if (len > 12) {
    is_scrolling_active = true; scroll_state = SCROLLING; scroll_index = 0; last_scroll_time = 0;
  } else {
    is_scrolling_active = false; char display_buffer[13];
    strncpy(display_buffer, sequence_string, len);
    for (int i = len; i < 12; i++) { display_buffer[i] = ' '; }
    display_buffer[12] = '\0'; lcd.setCursor(0, 1); lcd.print(display_buffer);
  }
}
void say_test_stripl() { lcd.setCursor(0, 0); lcd.print(F("Pro.Sca.Lin.    ")); say_tipro(); say_time(time_test[ch]); }
void say_up() { lcd.setCursor(0, 0); lcd.print(F("Count Up        ")); say_time(0); }
void say_down() { lcd.setCursor(0, 0); lcd.print(F("Count Down      ")); say_time(time_countdown[ch]); }
void say_dds() { lcd.setCursor(0, 0); lcd.print(F("Count Down 2    ")); say_time(time_dds[ch]); }
void say_test_stripf() { lcd.setCursor(0, 0); lcd.print(F("Pro.Sca.Fstop   ")); say_tipro(); say_time(time_fsttest[ch]); }
void say_fstopdown() { lcd.setCursor(0, 0); lcd.print(F("CountDown Fstop ")); say_time(time_fstdown[ch]); }
void say_fstopdown2() { lcd.setCursor(0, 0); lcd.print(F("CountDown Fstop2")); say_time(time_fstdown2[ch]); }
void say_burn() { lcd.setCursor(0, 0); lcd.print(F("Scher.Bruc.Fstop")); say_time(time_brn[ch]); }
void say_svfatt() {
  lcd.setCursor(0, 0); lcd.print(F("Svil. Fattoriale")); say_time(tsvfat_time[0][ch]);
  build_sequence_string(); int len = strlen(sequence_string);
  if (len > 12) {
    is_scrolling_active = true; scroll_state = SCROLLING; scroll_index = 0; last_scroll_time = 0;
  } else {
    is_scrolling_active = false; char display_buffer[13];
    strncpy(display_buffer, sequence_string, len);
    for (int i = len; i < 12; i++) { display_buffer[i] = ' '; }
    display_buffer[12] = '\0'; lcd.setCursor(0, 1); lcd.print(display_buffer);
  }
}
void say_timermode() {
  switch (timer_mode) {
    case FOCUS: say_free(); break;
    case MCDOWN: say_mcdown(); break;
    case PSCAL: say_test_stripl(); break;
    case CUP: say_up(); break;
    case CDOWN: say_down(); break;
    case CDOWN2: say_dds(); break;
    case PSCAF: say_test_stripf(); break;
    case CFSTOP: say_fstopdown(); break;
    case CFSTOP2: say_fstopdown2(); break;
    case MBFSTP: say_burn(); break;
    case SVFAT: say_svfatt(); break;
  }
}
void say_tipro() { lcd.setCursor(14, 0); lcd.print(provcomp ? F("PP") : F("CO")); }
void say_sound() { lcd.setCursor(12, 1); lcd.write(CHAR_NOTE); }
void say_mute() { lcd.setCursor(12, 1); lcd.print(F(" ")); }
void say_rel() { lcd.setCursor(13, 1); if (attrel[ch]) { lcd.write(CHAR_SPINA); } else { lcd.print(F(" ")); } }
void say_preoril() { lcd.setCursor(14, 1); if (preoril == LOW) { lcd.write(CHAR_FRECCIA_GIU); } else { lcd.write(CHAR_FRECCIA_SU); } }
void say_reset() { d7seg.write(D7SEG_CMD_RESET); d7seg.print(F("0000")); d7seg.write(D7SEG_CMD_CURSOR); d7seg.write(D7SEG_DOT_DECIMAL); }
void say_err() { d7seg.write(D7SEG_CMD_RESET); d7seg.print(F("Err ")); }
void say_ok() { d7seg.write(D7SEG_CMD_RESET); d7seg.print(F("Stor")); }
void say_reg() { d7seg.write(D7SEG_CMD_RESET); d7seg.print(F("End ")); }
void say_clr() { d7seg.write(D7SEG_CMD_RESET); d7seg.print(F("Clr ")); }
void say_rst() { d7seg.write(D7SEG_CMD_RESET); d7seg.print(F("RSt ")); }

void say_version() {
  lcd.setCursor(0, 0); lcd.print(F(" Timerino 3.0.8 "));
  lcd.setCursor(0, 1); lcd.print(F("   Buon Lavoro   "));
  d7seg.write(D7SEG_CMD_RESET); d7seg.print(F("0000")); delay(2000);
}

void update_icons() {
  if (mute) say_mute(); else say_sound();
  say_rel(); say_preoril();
}

void beep(int this_tone, int duration) { if (!mute) tone(PIN_BUZZER, this_tone, duration); }
void metronome(int ar_time) { if (currentState != STATE_IDLE && ar_time % 10 == 0) beep(TONE_DOWN, 80); }

void startBeepSequence(byte numBeeps, int tone, int duration) {
  if (beepSequenceActive) return;
  beepSequenceLength = numBeeps * 2 - 1; if (numBeeps == 1) beepSequenceLength = 1;
  currentBeepTone = tone; currentBeepDuration = duration;
  beepSequenceActive = true; beepStep = 1; lastBeepActionTime = millis();
  beep(currentBeepTone, currentBeepDuration);
}

void handleBeeps() {
  if (!beepSequenceActive) return;
  unsigned long currentTime = millis();
  if (beepStep >= beepSequenceLength) { beepSequenceActive = false; return; }
  if (beepStep % 2 == 1) {
    if (currentTime - lastBeepActionTime >= 150) { beepStep++; lastBeepActionTime = currentTime; }
  } else {
    beep(currentBeepTone, currentBeepDuration); beepStep++; lastBeepActionTime = currentTime;
  }
}

void metronomeSVFAT(int ar_time) {
  if (beepSequenceActive) return;
  bool isRunning = (currentState == STATE_SVFAT_COUNTDOWN || currentState == STATE_SVFAT_COUNTUP || currentState == STATE_SVFAT_FACTORIAL_DOWN);
  if (isRunning && ar_time % 300 == 0 && ar_time >= 300 && ar_time != 0) { beep(TONE_DOWN, 500); }
  else if (isRunning && ar_time % 150 == 0 && ar_time < 300 && ar_time != 0) { startBeepSequence(3, TONE_UP, 75); }
  else if (isRunning && ar_time % 100 == 0 && ar_time < 150 && ar_time != 0) { startBeepSequence(2, TONE_UP, 75); }
  else if (isRunning && ar_time % 10 == 0 && ar_time < 59 && ar_time != 0) { beep(TONE_DOWN, 80); }
}

void set_precis() {
  switch (iprec) {
    case 0: precis = 20; break; case 1: precis = 15; break; case 2: precis = 10; break;
    case 3: precis = 7; break; case 4: precis = 5; break; case 5: precis = 3; break;
    case 6: precis = 2; break; case 7: case 8: case 9: case 10: precis = 1; break;
    case 11: precis = 1; break; case 12: precis = 2; break; case 13: precis = 3; break;
    case 14: precis = 4; break; case 15: precis = 6; break; case 16: precis = 8; break;
    case 17: precis = 12; break; case 18: precis = 16; break; case 19: precis = 24; break;
    case 20: precis = 32; break; case 21: precis = 64; break;
  }
  say_precis();
}

void write_eeprom() {
  if (timer_mode == MCDOWN) {
    int start_address = EEPROM_ADDR_MCD_START + ((ch - 1) * EEPROM_CHANNEL_SIZE_MCD);
    for (int k = 0; k < 10; k++) {
      int current_address = start_address + (k * 2);
      if (k < imcd[ch]) { eeprom_write_word((uint16_t *)current_address, tmcd_time[k][ch]); }
      else { eeprom_write_word((uint16_t *)current_address, 0); }
      delay(5);
    }
  }
  if (timer_mode == SVFAT) {
    int start_address = EEPROM_ADDR_SVFAT_START + ((ch - 1) * EEPROM_CHANNEL_SIZE_SVFAT);
    for (int k = 0; k < 10; k++) {
      int current_address = start_address + (k * 2);
      if (k < isvfat[ch]) { eeprom_write_word((uint16_t *)current_address, tsvfat_time[k][ch]); }
      else { eeprom_write_word((uint16_t *)current_address, 0); }
      delay(5);
    }
  }
}

void load_eeprom() {
  brightness = eeprom_read_word((uint16_t *)EEPROM_ADDR_BRIGHTNESS);
  if (brightness == 0xFFFF || brightness > 100) brightness = 80;

  for (byte j = 1; j <= 3; j++) {
    imcd[j] = 0;
    isvfat[j] = 0;
    int mcd_start_address = EEPROM_ADDR_MCD_START + ((j - 1) * EEPROM_CHANNEL_SIZE_MCD);
    int svfat_start_address = EEPROM_ADDR_SVFAT_START + ((j - 1) * EEPROM_CHANNEL_SIZE_SVFAT);
    
    for (byte i = 0; i < 10; i++) {
      tmcd_time[i][j] = eeprom_read_word((uint16_t *)(mcd_start_address + i * 2));
      if (tmcd_time[i][j] > 0 && tmcd_time[i][j] != 0xFFFF) { imcd[j] = i + 1; }
      
      tsvfat_time[i][j] = eeprom_read_word((uint16_t *)(svfat_start_address + i * 2));
      if (tsvfat_time[i][j] > 0 && tsvfat_time[i][j] != 0xFFFF) { isvfat[j] = i + 1; }
    }
  }

  for (int index = 1; index <= 3; index++) {
    attrel[index] = eeprom_read_byte((uint8_t *)(EEPROM_ADDR_ATTREL_START + index));
    
    time_countdown[index] = 0;
    time_dds[index] = 0;
    time_fsttest[index] = 0;
    time_fstdown[index] = 0;
    time_fstdown2[index] = 0;
    time_brn[index] = 0;
    time_test[index] = 0;
  }

  byte preoril_val = eeprom_read_byte((uint8_t *)EEPROM_ADDR_PREORIL);
  if (preoril_val == 0xFF) {
    preoril = LOW;
  } else {
    preoril = preoril_val ? HIGH : LOW;
  }
  
  total_relay_seconds = eeprom_read_dword((uint32_t*)EEPROM_ADDR_HOUR_COUNTER);
  if (total_relay_seconds == 0xFFFFFFFF) {
    total_relay_seconds = 0;
  }
}