/*
  Timer per ingranditore camera oscura "Timerino_3.0.8"
  2025-11-02 Alessio Bortoletto
*/

//-----------------------------------------------------------------------------------------------------
// LIBRERIE
#include <Keypad.h>
#include <avr/eeprom.h>
#include <Wire.h>
#include <LiquidCrystal_I2C.h>
#include <SoftwareSerial.h>

//-----------------------------------------------------------------------------------------------------
// DEFINIZIONE DEGLI STATI DEL PROGRAMMA
enum ProgramState {
  STATE_IDLE, STATE_COUNTDOWN, STATE_MCDOWN, STATE_STOPWATCH, STATE_TEST_STRIP,
  STATE_MBFSTP_BURN, STATE_MBFSTP_EXPOSURE, STATE_SVFAT_COUNTUP, STATE_SVFAT_FACTORIAL_DOWN, STATE_SVFAT_COUNTDOWN,
  STATE_WAITING_FOR_SECOND_PRESS, STATE_HOUR_COUNTER_DISPLAY
};
ProgramState currentState = STATE_IDLE;

//-----------------------------------------------------------------------------------------------------
// CARATTERI PERSONALIZZATI LCD
uint8_t note[8] = {0x2, 0x3, 0x2, 0x2, 0xe, 0x1e, 0x1e, 0xc};
uint8_t spina[8] = {0xa, 0xa, 0x1f, 0x11, 0x11, 0xe, 0x4, 0x4};
uint8_t frecciasu[8] = {0x4, 0xe, 0x1f, 0x4, 0x4, 0x4, 0x4, 0x4};
uint8_t frecciagiu[8] = {0x4, 0x4, 0x4, 0x4, 0x4, 0x1f, 0xe, 0x4};

//-----------------------------------------------------------------------------------------------------
// COSTANTI
const byte buzzer = 10; const byte relay = 11; const byte mainbtn = 12; const byte offswi = 13;
const byte funzswi = A0; const byte precswi = A1; const byte disp1 = A2; const byte callsvf = A3;
const int tone_up = 600; const int tone_down = 300;
const byte ROWS = 4; const byte COLS = 4;

// --- NUOVA SEQUENZA MODALITA' ---
const byte FOCUS     =  1; // 01 Impostazioni e messa a fuoco
const byte MCDOWN    =  2; // 02 Conto alla rovescia multiplo
const byte PSCAL     =  3; // 03 Provino a scalare lineare
const byte CUP       =  4; // 04 Conto in avanti
const byte CDOWN     =  5; // 05 Conto alla rovescia
const byte CDOWN2    =  6; // 06 Conto alla rovescia con primo giro a vuoto
const byte PSCAF     =  7; // 07 Provino a scalare f-stop
const byte CFSTOP    =  8; // 08 Calcolatore f-stop
const byte CFSTOP2   =  9; // 09 Calcolatore f-stop con primo giro a vuoto
const byte MBFSTP    = 10; // 10 Scherma-Brucia f-stop
const byte SVFAT     = 11; // 11 Sviluppo stampa fattoriale

//-----------------------------------------------------------------------------------------------------
// VARIABILI GLOBALI
boolean isFocusRelayOn = false;
boolean beepSequenceActive = false; byte beepStep = 0; byte beepSequenceLength = 0;
unsigned long lastBeepActionTime = 0; int currentBeepTone = 0; int currentBeepDuration = 0;
int tsvfat_time[10][4]; byte isvfat[4] = {0, 0, 0, 0};
boolean mute = false; boolean attrel[4]; boolean provcomp = true;
byte timer_mode = 0; byte ch = 1; int brightness = 0; int preoril;
byte ncifra = 0; int time = 0; int time_succ; int appo_time; int time_burn; int initial_time_burn = 0;
int time_countdown[4]; int time_dds[4]; int time_fsttest[4]; int time_fstdown[4];
int time_fstdown2[4]; int time_brn[4]; int time_test[4]; int time_svfatt[4];
int precis = 1; int iprec = 0; int last_timer_mod = -1; int last_iprec = -1;
byte sca = 0; int svfatInputIndex = 0;
int tmcd_time[10][4]; byte imcd[4] = {0, 0, 0, 0};
long last_time = 0; float mult = 1.0; char _buffer[17];
char keys[ROWS][COLS] = { {'1', '2', '3', 'A'}, {'4', '5', '6', 'B'}, {'7', '8', '9', 'C'}, {'*', '0', '#', 'D'} };
byte rowPins[ROWS] = {6, 7, 8, 9}; byte colPins[COLS] = {2, 3, 4, 5};
bool isDodgeOperation = false;
bool isTimerPaused = false;

// --- VARIABILI PER SCORRIMENTO TESTO SEQUENZA ---
char sequence_string[100];
bool is_scrolling_active = false;
int scroll_index = 0;
unsigned long last_scroll_time = 0;
const int scroll_speed_ms = 400;
const int scroll_pause_duration_ms = 1000;

enum ScrollState { SCROLLING, PAUSED, STATIC_DISPLAY };
ScrollState scroll_state = SCROLLING;
unsigned long state_timer = 0;
const int scroll_static_duration_ms = 3000;

// --- VARIABILI CONTA-ORE LAMPADA ---
unsigned long total_relay_seconds = 0;
unsigned long relay_on_time = 0;
const int eeprom_hour_counter_address = 200;

//-----------------------------------------------------------------------------------------------------
// OGGETTI
Keypad keypad = Keypad( makeKeymap(keys), rowPins, colPins, ROWS, COLS);
SoftwareSerial d7seg(255, disp1);
LiquidCrystal_I2C lcd(0x27, 16, 2);

//-----------------------------------------------------------------------------------------------------
// PROTOTIPI DI FUNZIONE
void say_time(int ar_time); void init_timermode(); void say_timermode(); void set_precis();
void say_ch(); void say_clr(); void say_reset(); void say_clearlcd();
void say_sound(); void say_rel(); void say_preoril(); void beep(int this_tone, int duration);
void metronome(int ar_time); void metronomeSVFAT(int ar_time); void load_eeprom();
void write_eeprom(); void say_tipro(); void readKeypad(); void readSwitches();
void say_precis(); void say_ok(); void say_err(); void say_reg(); void say_rst();
void startBeepSequence(byte numBeeps, int tone, int duration);
void say_version();
void handleIdleState(bool buttonPressed);
void handleCountdownState(bool buttonPressed);
void handleMcdownState(bool buttonPressed);
void handleSvfCountUpState(bool buttonPressed);
void handleSvfFactorialDownState(bool buttonPressed);
void handleSvfCountdownState(bool buttonPressed);
void handleStopwatchState(bool buttonPressed);
void handleTestStripState(bool buttonPressed);
void handleWaitingForSecondPressState(bool buttonPressed);
void handleMbFstpBurnState(bool buttonPressed);
void handleMbFstpExposureState(bool buttonPressed);
void handleHourCounterDisplayState();
void handleBeeps();
void clearInputTime();
void say_mute();
void reset_mcdown_channel(byte channel);
void reset_svfat_channel(byte channel);
void reset_hour_counter();
void update_icons();
void build_sequence_string();
void handle_sequence_display();
void clear_second_line_content();
void turn_relay_on();
void turn_relay_off();
void save_channel_time(byte channel_to_save); // <-- PROTOTIPO NUOVA FUNZIONE


//-----------------------------------------------------------------------------------------------------
// SETUP
void setup() {
  Serial.begin(9600);
  pinMode(funzswi, INPUT); pinMode(precswi, INPUT); pinMode(disp1, OUTPUT); pinMode(buzzer, OUTPUT);
  pinMode(relay, OUTPUT); digitalWrite(relay, HIGH);
  pinMode(mainbtn, INPUT);
  pinMode(offswi, INPUT_PULLUP); pinMode(callsvf, INPUT_PULLUP);
  keypad.setDebounceTime(50);
  load_eeprom();
  lcd.init(); lcd.backlight();
  lcd.createChar(1, note); lcd.createChar(2, spina); lcd.createChar(3, frecciasu); lcd.createChar(4, frecciagiu);
  d7seg.begin(9600); d7seg.write(0x76); d7seg.write(0x7A); d7seg.write((byte) brightness);
  say_version();
  say_reset(); say_clearlcd();
  currentState = STATE_IDLE;
  init_timermode();
}

//-----------------------------------------------------------------------------------------------------
// LOOP PRINCIPALE A STATI
void loop() {
  handle_sequence_display();
  handleBeeps();
  static unsigned long lastButtonCheck = 0;
  static bool lastButtonState = false;
  bool buttonPressed = false;
  if (millis() - lastButtonCheck > 50) {
    bool currentButtonState = (digitalRead(mainbtn) == HIGH);

    if (preoril == LOW) {
      if (currentButtonState && !lastButtonState) {
        buttonPressed = true;
      }
    } else {
      if (!currentButtonState && lastButtonState) {
        buttonPressed = true;
      }
    }
    
    lastButtonState = currentButtonState;
    lastButtonCheck = millis();
  }

  switch (currentState) {
    case STATE_IDLE: handleIdleState(buttonPressed); break;
    case STATE_COUNTDOWN: handleCountdownState(buttonPressed); break;
    case STATE_MCDOWN: readKeypad(); handleMcdownState(buttonPressed); break;
    case STATE_STOPWATCH: handleStopwatchState(buttonPressed); break;
    case STATE_TEST_STRIP: handleTestStripState(buttonPressed); break;
    case STATE_MBFSTP_BURN: handleMbFstpBurnState(buttonPressed); break;
    case STATE_MBFSTP_EXPOSURE: handleMbFstpExposureState(buttonPressed); break;
    case STATE_SVFAT_COUNTUP: handleSvfCountUpState(buttonPressed); break;
    case STATE_SVFAT_FACTORIAL_DOWN: readKeypad(); handleSvfFactorialDownState(buttonPressed); break;
    case STATE_SVFAT_COUNTDOWN: readKeypad(); handleSvfCountdownState(buttonPressed); break;
    case STATE_WAITING_FOR_SECOND_PRESS: readSwitches(); readKeypad(); handleWaitingForSecondPressState(buttonPressed); break;
    case STATE_HOUR_COUNTER_DISPLAY: handleHourCounterDisplayState(); break;
  }
}

//-----------------------------------------------------------------------------------------------------
// FUNZIONI DI GESTIONE DEGLI STATI

void handleIdleState(bool buttonPressed) {
  readSwitches();
  readKeypad();
  if (buttonPressed) {
    switch (timer_mode) {
      case FOCUS:
        if (attrel[ch]) {
          if (isFocusRelayOn) {
            turn_relay_off(); isFocusRelayOn = false;
          } else {
            turn_relay_on(); isFocusRelayOn = true;
          }
        }
        break;
      case CUP:
        last_time = millis(); if (attrel[ch]) { turn_relay_on(); }
        currentState = STATE_STOPWATCH; break;
      case MCDOWN:
        if (imcd[ch] > 0 && tmcd_time[0][ch] > 0) {
          time = tmcd_time[0][ch];
          sca = 0;
          last_time = millis();
          if (attrel[ch]) {
            turn_relay_on();
          }
          isTimerPaused = false;
          currentState = STATE_MCDOWN;
        }
        break;
      case CDOWN: case CFSTOP:
        if (time > 0) {
          appo_time = time; last_time = millis();
          if (attrel[ch]) { turn_relay_on(); }
          currentState = STATE_COUNTDOWN;
        }
        break;
      case CDOWN2: case CFSTOP2:
        if (time > 0) {
          appo_time = time;
          if (attrel[ch]) { turn_relay_on(); }
          currentState = STATE_WAITING_FOR_SECOND_PRESS;
        }
        break;
      case PSCAL: case PSCAF:
        appo_time = time; if (time < 10) { time = 10; appo_time = 10; }
        time_succ = appo_time; time = 0; last_time = millis();
        if (attrel[ch]) { turn_relay_on(); }
        currentState = STATE_TEST_STRIP;
        break;
      case MBFSTP:
        if (time > 0) {
          appo_time = time;
          if (time_burn > 0) {
            if (attrel[ch]) { turn_relay_on(); }
            last_time = millis(); currentState = STATE_MBFSTP_BURN;
          } else {
            if (attrel[ch]) { turn_relay_on(); }
            last_time = millis(); currentState = STATE_MBFSTP_EXPOSURE;
          }
        }
        break;
      case SVFAT:
        if (isvfat[ch] > 1) {
          time = 0; last_time = millis(); isTimerPaused = false;
          currentState = STATE_SVFAT_COUNTUP;
        }
        break;
    }
  }
}

void handleCountdownState(bool buttonPressed) {
  if (currentState != STATE_COUNTDOWN) return;
  if (buttonPressed) {
    if (attrel[ch]) { turn_relay_off(); }
    time = appo_time; say_time(time); currentState = STATE_IDLE; return;
  }
  if (millis() - last_time >= 100) {
    last_time = millis(); time--; metronome(time); say_time(time);
    if (time <= 0) {
      if (attrel[ch]) { turn_relay_off(); }
      startBeepSequence(3, tone_down, 150);
      time = appo_time; say_time(time); currentState = STATE_IDLE;
    }
  }
}

void handleMcdownState(bool buttonPressed) {
  if (currentState != STATE_MCDOWN) return;

  if (buttonPressed) {
    isTimerPaused = !isTimerPaused;
    if (isTimerPaused) {
      if (attrel[ch]) { turn_relay_off(); }
    } else {
      if (attrel[ch]) { turn_relay_on(); }
      last_time = millis();
    }
    return;
  }

  if (!isTimerPaused) {
    if (millis() - last_time >= 100) {
      last_time = millis();
      time--;
      say_time(time);
      if (time <= 0) {
        startBeepSequence(3, tone_up, 75);
        sca++;
        if (sca < imcd[ch]) {
          time = tmcd_time[sca][ch];
          say_time(time);
        } else {
          if(attrel[ch]) { turn_relay_off(); }
          time = tmcd_time[0][ch];
          say_time(time);
          currentState = STATE_IDLE;
        }
      }
    }
  }
}

void handleSvfCountUpState(bool buttonPressed) {
  if (buttonPressed) {
    int calculated_time = time * tsvfat_time[0][ch] - time;
    if (calculated_time < 0) calculated_time = 0;
    time = calculated_time; last_time = millis(); sca = 1;
    currentState = STATE_SVFAT_FACTORIAL_DOWN; return;
  }
  if (millis() - last_time >= 100) {
    last_time = millis(); time++; metronome(time); say_time(time);
  }
}

void handleSvfFactorialDownState(bool buttonPressed) {
  if (currentState != STATE_SVFAT_FACTORIAL_DOWN) return;
  if (buttonPressed) { isTimerPaused = !isTimerPaused; if (!isTimerPaused) { last_time = millis(); } }
  if (!isTimerPaused) {
    if (millis() - last_time >= 100) {
      last_time = millis(); time--; metronomeSVFAT(time); say_time(time);
      if (time <= 0) {
        startBeepSequence(3, tone_up, 75);
        if (sca < isvfat[ch]) {
          currentState = STATE_SVFAT_COUNTDOWN; time = tsvfat_time[sca][ch]; say_time(time);
        } else {
          currentState = STATE_IDLE; time = tsvfat_time[0][ch]; say_time(time);
        }
      }
    }
  }
}

void handleSvfCountdownState(bool buttonPressed) {
  if (currentState != STATE_SVFAT_COUNTDOWN) return;
  if (buttonPressed) { isTimerPaused = !isTimerPaused; if (!isTimerPaused) { last_time = millis(); } }
  if (!isTimerPaused) {
    if (millis() - last_time >= 100) {
      last_time = millis(); time--; metronomeSVFAT(time); say_time(time);
      if (time <= 0) {
        startBeepSequence(3, tone_up, 75); sca++;
        if (sca < isvfat[ch]) {
          time = tsvfat_time[sca][ch]; say_time(time);
        } else {
          currentState = STATE_IDLE; time = tsvfat_time[0][ch]; say_time(time);
        }
      }
    }
  }
}

void handleStopwatchState(bool buttonPressed) {
  if (buttonPressed) {
    if (attrel[ch]) { turn_relay_off(); }
    currentState = STATE_IDLE; return;
  }
  if (millis() - last_time >= 100) {
    last_time = millis(); time++; metronome(time); say_time(time);
  }
}

void handleTestStripState(bool buttonPressed) {
  if (buttonPressed) {
    if (attrel[ch]) { turn_relay_off(); }
    time = appo_time; say_time(time); currentState = STATE_IDLE; return;
  }
  if (millis() - last_time >= 100) {
    last_time = millis(); time++; say_time(time);
    if (time >= time_succ) {
      beep(tone_up, 150);
      if (timer_mode == PSCAF) {
        mult = pow(2.0, (1.0 / precis)); time_succ = round((float)time_succ * mult);
      } else {
        time_succ = time_succ + (precis * 10);
      }
      if (provcomp) {
        if (attrel[ch]) { turn_relay_off(); }
        say_time(time_succ); currentState = STATE_WAITING_FOR_SECOND_PRESS;
      }
    }
    if (time_succ > time && time_succ - time <= 3) { beep(tone_up, 50); }
  }
}

void handleWaitingForSecondPressState(bool buttonPressed) {
  if (buttonPressed) {
    if (attrel[ch]) {
      turn_relay_on();
    }
    last_time = millis();

    if ((timer_mode == PSCAL || timer_mode == PSCAF) && provcomp) {
      time = 0;
      currentState = STATE_TEST_STRIP;
    } else if (timer_mode == MBFSTP && isDodgeOperation) {
      currentState = STATE_MBFSTP_EXPOSURE;
    } else {
      currentState = STATE_COUNTDOWN;
    }
  }
}

void handleMbFstpBurnState(bool buttonPressed) {
  if (buttonPressed) {
    if (attrel[ch]) { turn_relay_off(); }
    time = appo_time;
    say_time(time);
    currentState = STATE_IDLE;
    return;
  }
  if (millis() - last_time >= 100) {
    last_time = millis();
    time_burn--;
    metronome(time_burn);
    say_time(time_burn);
    if (time_burn <= 0) {
      if (attrel[ch]) { turn_relay_off(); }

      if (isDodgeOperation) {
        time = appo_time - initial_time_burn;
        appo_time = time;
      } else {
        time = appo_time;
      }
      say_time(time);
      currentState = STATE_WAITING_FOR_SECOND_PRESS;
    }
  }
}

void handleMbFstpExposureState(bool buttonPressed) {
  handleCountdownState(buttonPressed);
}

void handleHourCounterDisplayState() {
  readKeypad();
  if (millis() - last_time > 4000) {
    currentState = STATE_IDLE;
    init_timermode();
  }
}

//-----------------------------------------------------------------------------------------------------
// FUNZIONI DI SUPPORTO
//-----------------------------------------------------------------------------------------------------

/**
 * Salva il valore 'time' globale nell'array del canale specificato,
 * in base alla modalità timer attiva. Questo serve per memorizzare
 * temporaneamente il tempo quando si cambia canale.
 */
void save_channel_time(byte channel_to_save) {
  switch (timer_mode) {
    case CDOWN:   time_countdown[channel_to_save] = time; break;
    case CDOWN2:  time_dds[channel_to_save] = time; break;
    case PSCAL:   time_test[channel_to_save] = time; break;
    case PSCAF:   time_fsttest[channel_to_save] = time; break;
    case CFSTOP:  time_fstdown[channel_to_save] = time; break;
    case CFSTOP2: time_fstdown2[channel_to_save] = time; break;
    case MBFSTP:  time_brn[channel_to_save] = time; break;
    // Gli altri modi (MCDOWN, SVFAT, FOCUS, CUP) non necessitano di questo salvataggio
    // poichè gestiscono i tempi in modo diverso (EEPROM o non memorizzati).
  }
}

void turn_relay_on() {
  if (digitalRead(relay) == HIGH) {
    digitalWrite(relay, LOW);
    relay_on_time = millis();
  }
}

void turn_relay_off() {
  if (digitalRead(relay) == LOW) {
    unsigned long duration_ms = millis() - relay_on_time;
    total_relay_seconds += duration_ms / 1000;
    digitalWrite(relay, HIGH);
  }
}

void clear_second_line_content() {
  lcd.setCursor(0, 1);
  lcd.print("                ");
}

void build_sequence_string() {
  sequence_string[0] = '\0';
  char temp_buffer[8];
  byte num_steps = 0;
  int start_index = 0;
  int (*time_array)[4] = NULL;

  if (timer_mode == MCDOWN) {
    num_steps = imcd[ch]; time_array = tmcd_time; start_index = 0;
  } else if (timer_mode == SVFAT) {
    num_steps = isvfat[ch]; time_array = tsvfat_time; start_index = 1;
  }

  if (num_steps > start_index) {
    for (int i = start_index; i < num_steps; i++) {
      int seconds = time_array[i][ch] / 10;
      sprintf(temp_buffer, "%d", seconds);
      strcat(sequence_string, temp_buffer);
      if (i < num_steps - 1) { strcat(sequence_string, "-"); }
    }
  }
}

void handle_sequence_display() {
  if (!is_scrolling_active) return;
  int len = strlen(sequence_string);

  switch (scroll_state) {
    case SCROLLING:
      if (millis() - last_scroll_time > scroll_speed_ms) {
        last_scroll_time = millis();
        char display_buffer[13];
        memset(display_buffer, ' ', 12); display_buffer[12] = '\0';
        int chars_to_copy = min(len - scroll_index, 12);
        if (chars_to_copy > 0) { strncpy(display_buffer, sequence_string + scroll_index, chars_to_copy); }
        lcd.setCursor(0, 1); lcd.print(display_buffer);
        scroll_index++;
        if (scroll_index > len) {
          scroll_state = PAUSED; state_timer = millis();
          lcd.setCursor(0, 1); lcd.print("            ");
        }
      }
      break;
    case PAUSED:
      if (millis() - state_timer > scroll_pause_duration_ms) {
        scroll_state = STATIC_DISPLAY; state_timer = millis();
        char static_buffer[13];
        strncpy(static_buffer, sequence_string, 12); static_buffer[12] = '\0';
        for (int i = strlen(static_buffer); i < 12; i++) { static_buffer[i] = ' '; }
        lcd.setCursor(0, 1); lcd.print(static_buffer);
      }
      break;
    case STATIC_DISPLAY:
      if (millis() - state_timer > scroll_static_duration_ms) {
        scroll_state = SCROLLING; scroll_index = 0; last_scroll_time = millis();
      }
      break;
  }
}

void clearInputTime() {
  time = 0;               // Imposta il tempo globale a 0
  ncifra = 0;             // Azzera il contatore cifre
  save_channel_time(ch);  // Salva questo 0 nell'array del canale corrente
  init_timermode();       // Ricarica la modalità (ora caricherà 0)
  currentState = STATE_IDLE;
}

void reset_mcdown_channel(byte channel) {
  imcd[channel] = 0;
  for (int i = 0; i < 10; i++) { tmcd_time[i][channel] = 0; }
  time = 0; ncifra = 0; say_time(time); write_eeprom(); init_timermode();
}

void reset_svfat_channel(byte channel) {
  isvfat[channel] = 0;
  for (int i = 0; i < 10; i++) { tsvfat_time[i][channel] = 0; }
  time = 0; ncifra = 0; say_time(time); write_eeprom(); init_timermode();
}

void reset_hour_counter() {
  total_relay_seconds = 0;
  eeprom_write_dword((uint32_t*)eeprom_hour_counter_address, total_relay_seconds);
  lcd.clear();
  lcd.setCursor(0, 0);
  lcd.print("Contaore");
  lcd.setCursor(0, 1);
  lcd.print("Azzerato!");
  delay(1500);
}

void readKeypad() {
  char key = keypad.getKey();
  if (!key) return;

  switch (key) {
    case 'A': 
      save_channel_time(ch); // <-- MODIFICA: Salva tempo canale corrente
      ch = 1; 
      beep(tone_down, 200); 
      init_timermode(); 
      break;
    case 'B': 
      save_channel_time(ch); // <-- MODIFICA: Salva tempo canale corrente
      ch = 2; 
      beep(tone_down, 200); 
      init_timermode(); 
      break;
    case 'C': 
      save_channel_time(ch); // <-- MODIFICA: Salva tempo canale corrente
      ch = 3; 
      beep(tone_down, 200); 
      init_timermode(); 
      break;
    
    case 'D':
      if (timer_mode == FOCUS) {
        say_clr();
        delay(250);
        reset_hour_counter();
        init_timermode();
        currentState = STATE_IDLE;
      }
      else if ((timer_mode == MCDOWN || timer_mode == SVFAT) && currentState != STATE_IDLE) {
        say_rst();
        delay(250);
        isTimerPaused = false;
        currentState = STATE_IDLE;
        init_timermode();
      } else {
        say_clr();
        delay(250);
        if (timer_mode == MCDOWN) {
          reset_mcdown_channel(ch);
        } else if (timer_mode == SVFAT) {
          reset_svfat_channel(ch);
        } else {
          clearInputTime();
        }
      }
      break;

    case '*':
      if (timer_mode == FOCUS) {
        if (brightness > 0) {
          brightness -= 10; d7seg.write(0x7A); d7seg.write((byte)brightness);
          eeprom_write_word((uint16_t *)1, brightness);
        }
      } else if (timer_mode == PSCAL || timer_mode == PSCAF) {
        provcomp = false; say_tipro();
      } else if (timer_mode == MCDOWN) {
        if (time != 0) {
          if (imcd[ch] < 10) {
            tmcd_time[imcd[ch]][ch] = time; say_ok(); delay(500);
            time = 0; ncifra = 0; say_time(time); imcd[ch]++;
          } else { say_err(); delay(500); }
        }
      } else if (timer_mode == SVFAT) {
        if (time != 0) {
          if (isvfat[ch] < 10) {
            tsvfat_time[isvfat[ch]][ch] = time; say_ok(); delay(500);
            time = 0; ncifra = 0; say_time(time); isvfat[ch]++;
          } else { say_err(); delay(500); }
        }
      } else if (timer_mode == CDOWN || timer_mode == CDOWN2) {
        int step = precis * 10; time -= step; if (time < 0) time = 0; say_time(time);
      } else if (timer_mode == CFSTOP || timer_mode == CFSTOP2) {
        float mult = pow(2.0, (1.0 / precis)); time = round((float)time / mult);
        if (time < 0) time = 0; say_time(time);
      } else if (timer_mode == MBFSTP) {
        isDodgeOperation = true; mult = 1.0 / precis;
        time_burn = round((float)time * mult); initial_time_burn = time_burn; say_time(time_burn);
      }
      break;
    case '#':
      if (timer_mode == FOCUS) {
        if (brightness < 100) {
          brightness += 10; d7seg.write(0x7A); d7seg.write((byte)brightness);
          eeprom_write_word((uint16_t *)1, brightness);
        }
      } else if (timer_mode == PSCAL || timer_mode == PSCAF) {
        provcomp = true; say_tipro();
      } else if (timer_mode == MCDOWN) {
        if (imcd[ch] != 0 || time != 0) {
          if (time != 0) {
            if (imcd[ch] < 10) { tmcd_time[imcd[ch]][ch] = time; imcd[ch]++; }
          }
          write_eeprom(); say_reg(); delay(500);
          time = tmcd_time[0][ch]; say_time(time); say_mcdown();
        }
      } else if (timer_mode == SVFAT) {
        if (isvfat[ch] != 0 || time != 0) {
          if (time != 0) {
            if (isvfat[ch] < 10) { tsvfat_time[isvfat[ch]][ch] = time; isvfat[ch]++; }
          }
          write_eeprom(); say_reg(); delay(500);
          time = tsvfat_time[0][ch]; say_time(time); say_svfatt();
        }
      } else if (timer_mode == CDOWN || timer_mode == CDOWN2) {
        int step = precis * 10; time += step; if (time > 9999) time = 9999; say_time(time);
      } else if (timer_mode == CFSTOP || timer_mode == CFSTOP2) {
        float mult = pow(2.0, (1.0 / precis)); time = round((float)time * mult);
        if (time > 9999) time = 9999; say_time(time);
      } else if (timer_mode == MBFSTP) {
        isDodgeOperation = false; mult = 1.0 / precis;
        time_burn = round((float)time * mult); initial_time_burn = time_burn; say_time(time_burn);
      }
      break;
    default:
      if (isdigit(key)) {
        if (timer_mode == FOCUS) {
          switch (key) {
            case '0':
              mute = !mute; beep(tone_down, 200); update_icons(); break;
            case '1':
              attrel[ch] = !attrel[ch]; eeprom_write_byte((uint8_t *)(83 + ch), attrel[ch]);
              beep(tone_down, 200); update_icons(); break;
            case '2':
              preoril = (preoril == LOW) ? HIGH : LOW; eeprom_write_byte((uint8_t *)88, (preoril == HIGH));
              beep(tone_down, 200); update_icons(); break;
            case '3':
              char buffer[17];
              unsigned long hours = total_relay_seconds / 3600;
              unsigned int minutes = (total_relay_seconds % 3600) / 60;
              lcd.clear();
              lcd.setCursor(0, 0);
              lcd.print("Uso Lampada:");
              sprintf(buffer, "%lu h %u min", hours, minutes);
              lcd.setCursor(0, 1);
              lcd.print(buffer);
              currentState = STATE_HOUR_COUNTER_DISPLAY;
              last_time = millis();
              break;
          }
        } else if (timer_mode != CUP) {
          if (ncifra == 0) time = 0;
          if (time < 1000) { time = time * 10 + (key - '0'); }
          ncifra++; say_time(time);
        }
      }
      break;
  }
}

void readSwitches() {
  int swistate = digitalRead(offswi);
  int svfstate = digitalRead(callsvf);
  int new_timer_mode;

  if (swistate == LOW && svfstate == LOW) {
    new_timer_mode = ((analogRead(funzswi) + 47) * 11 / 1023) + 1;
  } else {
    new_timer_mode = (svfstate == LOW) ? SVFAT : FOCUS;
  }

  if (new_timer_mode != timer_mode) {
    eeprom_write_dword((uint32_t*)eeprom_hour_counter_address, total_relay_seconds);
    if (timer_mode == FOCUS && isFocusRelayOn) { turn_relay_off(); isFocusRelayOn = false; }
    
    // Salva il tempo PRIMA di cambiare modalità, se la modalità lo richiede
    save_channel_time(ch);
    
    timer_mode = new_timer_mode; 
    init_timermode(); 
    currentState = STATE_IDLE;
  }

  int p_val_raw = (analogRead(precswi) + 47) * 11 / 1023;
  int new_iprec = (timer_mode == PSCAF || timer_mode == CFSTOP || timer_mode == CFSTOP2 || timer_mode == MBFSTP) ? p_val_raw + 11 : p_val_raw;

  if (new_iprec != iprec) { iprec = new_iprec; set_precis(); }
}

void say_time(int ar_time) {
  d7seg.write(0x77);
  bool isRunning = (currentState != STATE_IDLE && currentState != STATE_WAITING_FOR_SECOND_PRESS);
  if (timer_mode == SVFAT) {
    d7seg.write((uint8_t)((!isRunning && (isvfat[ch] == 0 || ar_time == tsvfat_time[0][ch])) ? 0x00 : 0x04));
  } else { d7seg.write((uint8_t)0x04); }
  sprintf(_buffer, "%04i", ar_time); d7seg.write(_buffer);
}

void init_timermode() {
  clear_second_line_content();
  ncifra = 0; is_scrolling_active = false;
  switch (timer_mode) {
    case CDOWN:   time = time_countdown[ch]; break;
    case CDOWN2:  time = time_dds[ch]; break;
    case PSCAL:   time = time_test[ch]; break;
    case PSCAF:   time = time_fsttest[ch]; break;
    case CFSTOP:  time = time_fstdown[ch]; break;
    case CFSTOP2: time = time_fstdown2[ch]; break;
    case MBFSTP:  time = time_brn[ch]; break;
    case MCDOWN:  time = tmcd_time[0][ch]; sca = 0; break;
    case FOCUS:   time = 0; break;
    case SVFAT:   time = tsvfat_time[0][ch]; sca = 0; break;
    case CUP:     time = 0; break;
  }
  say_timermode(); set_precis(); say_time(time); update_icons(); say_ch();
}

void say_clearlcd() { lcd.clear(); }
void say_ch() { lcd.setCursor(15, 1); lcd.print((char)('A' + ch - 1)); }

void say_precis() {
  if (timer_mode == PSCAL || timer_mode == CDOWN || timer_mode == CDOWN2 ||
      timer_mode == PSCAF || timer_mode == CFSTOP || timer_mode == CFSTOP2 ||
      timer_mode == MBFSTP) {
    switch (iprec) {
      case 0: lcd.setCursor(0, 1); lcd.print("Pr:20s  "); break;
      case 1: lcd.setCursor(0, 1); lcd.print("Pr:15s  "); break;
      case 2: lcd.setCursor(0, 1); lcd.print("Pr:10s  "); break;
      case 3: lcd.setCursor(0, 1); lcd.print("Pr: 7s  "); break;
      case 4: lcd.setCursor(0, 1); lcd.print("Pr: 5s  "); break;
      case 5: lcd.setCursor(0, 1); lcd.print("Pr: 3s  "); break;
      case 6: lcd.setCursor(0, 1); lcd.print("Pr: 2s  "); break;
      case 7: case 8: case 9: case 10: lcd.setCursor(0, 1); lcd.print("Pr: 1s  "); break;
      case 11: lcd.setCursor(0, 1); lcd.print("Pr:1/1  "); break;
      case 12: lcd.setCursor(0, 1); lcd.print("Pr:1/2  "); break;
      case 13: lcd.setCursor(0, 1); lcd.print("Pr:1/3  "); break;
      case 14: lcd.setCursor(0, 1); lcd.print("Pr:1/4  "); break;
      case 15: lcd.setCursor(0, 1); lcd.print("Pr:1/6  "); break;
      case 16: lcd.setCursor(0, 1); lcd.print("Pr:1/8  "); break;
      case 17: lcd.setCursor(0, 1); lcd.print("Pr:1/12 "); break;
      case 18: lcd.setCursor(0, 1); lcd.print("Pr:1/16 "); break;
      case 19: lcd.setCursor(0, 1); lcd.print("Pr:1/24 "); break;
      case 20: lcd.setCursor(0, 1); lcd.print("Pr:1/32 "); break;
      case 21: lcd.setCursor(0, 1); lcd.print("Pr:1/64 "); break;
    }
  }
}

void say_free() {
  lcd.setCursor(0, 0); lcd.print("Focus      SetUp");
}
void say_mcdown() {
  lcd.setCursor(0, 0); lcd.print("Multi Count Down"); say_time(tmcd_time[0][ch]);
  build_sequence_string(); int len = strlen(sequence_string);
  if (len > 12) {
    is_scrolling_active = true; scroll_state = SCROLLING; scroll_index = 0; last_scroll_time = 0;
  } else {
    is_scrolling_active = false; char display_buffer[13];
    strncpy(display_buffer, sequence_string, len);
    for (int i = len; i < 12; i++) { display_buffer[i] = ' '; }
    display_buffer[12] = '\0'; lcd.setCursor(0, 1); lcd.print(display_buffer);
  }
}
void say_test_stripl() { lcd.setCursor(0, 0); lcd.print("Pro.Sca.Lin.    "); say_tipro(); say_time(time_test[ch]); }
void say_up() { lcd.setCursor(0, 0); lcd.print("Count Up        "); say_time(0); }
void say_down() { lcd.setCursor(0, 0); lcd.print("Count Down      "); say_time(time_countdown[ch]); }
void say_dds() { lcd.setCursor(0, 0); lcd.print("Count Down 2    "); say_time(time_dds[ch]); }
void say_test_stripf() { lcd.setCursor(0, 0); lcd.print("Pro.Sca.Fstop   "); say_tipro(); say_time(time_fsttest[ch]); }
void say_fstopdown() { lcd.setCursor(0, 0); lcd.print("CountDown Fstop "); say_time(time_fstdown[ch]); }
void say_fstopdown2() { lcd.setCursor(0, 0); lcd.print("CountDown Fstop2"); say_time(time_fstdown2[ch]); }
void say_burn() { lcd.setCursor(0, 0); lcd.print("Scher.Bruc.Fstop"); say_time(time_brn[ch]); }
void say_svfatt() {
  lcd.setCursor(0, 0); lcd.print("Svil. Fattoriale"); say_time(tsvfat_time[0][ch]);
  build_sequence_string(); int len = strlen(sequence_string);
  if (len > 12) {
    is_scrolling_active = true; scroll_state = SCROLLING; scroll_index = 0; last_scroll_time = 0;
  } else {
    is_scrolling_active = false; char display_buffer[13];
    strncpy(display_buffer, sequence_string, len);
    for (int i = len; i < 12; i++) { display_buffer[i] = ' '; }
    display_buffer[12] = '\0'; lcd.setCursor(0, 1); lcd.print(display_buffer);
  }
}
void say_timermode() {
  switch (timer_mode) {
    case FOCUS: say_free(); break;
    case MCDOWN: say_mcdown(); break;
    case PSCAL: say_test_stripl(); break;
    case CUP: say_up(); break;
    case CDOWN: say_down(); break;
    case CDOWN2: say_dds(); break;
    case PSCAF: say_test_stripf(); break;
    case CFSTOP: say_fstopdown(); break;
    case CFSTOP2: say_fstopdown2(); break;
    case MBFSTP: say_burn(); break;
    case SVFAT: say_svfatt(); break;
  }
}
void say_tipro() { lcd.setCursor(14, 0); lcd.print(provcomp ? "PP" : "CO"); }
void say_sound() { lcd.setCursor(12, 1); lcd.write(1); }
void say_mute() { lcd.setCursor(12, 1); lcd.print(" "); }
void say_rel() { lcd.setCursor(13, 1); if (attrel[ch]) { lcd.write(2); } else { lcd.print(" "); } }
void say_preoril() { lcd.setCursor(14, 1); if (preoril == LOW) { lcd.write(4); } else { lcd.write(3); } } // Logica icone invertita e corretta
void say_reset() { d7seg.write(0x76); d7seg.print("0000"); d7seg.write(0x77); d7seg.write(0b00000100); }
void say_err() { d7seg.write(0x76); d7seg.print("Err "); }
void say_ok() { d7seg.write(0x76); d7seg.print("Stor"); }
void say_reg() { d7seg.write(0x76); d7seg.print("End "); }
void say_clr() { d7seg.write(0x76); d7seg.print("Clr "); }
void say_rst() { d7seg.write(0x76); d7seg.print("RSt "); }

void say_version() {
  lcd.setCursor(0, 0); lcd.print(" Timerino 3.0.8 ");
  lcd.setCursor(0, 1); lcd.print("   Buon Lavoro   ");
  d7seg.write(0x76); d7seg.print("0000"); delay(2000);
}

void update_icons() {
  if (mute) say_mute(); else say_sound();
  say_rel(); say_preoril();
}

void beep(int this_tone, int duration) { if (!mute) tone(buzzer, this_tone, duration); }
void metronome(int ar_time) { if (currentState != STATE_IDLE && ar_time % 10 == 0) beep(tone_down, 80); }

void startBeepSequence(byte numBeeps, int tone, int duration) {
  if (beepSequenceActive) return;
  beepSequenceLength = numBeeps * 2 - 1; if (numBeeps == 1) beepSequenceLength = 1;
  currentBeepTone = tone; currentBeepDuration = duration;
  beepSequenceActive = true; beepStep = 1; lastBeepActionTime = millis();
  beep(currentBeepTone, currentBeepDuration);
}

void handleBeeps() {
  if (!beepSequenceActive) return;
  unsigned long currentTime = millis();
  if (beepStep >= beepSequenceLength) { beepSequenceActive = false; return; }
  if (beepStep % 2 == 1) {
    if (currentTime - lastBeepActionTime >= 150) { beepStep++; lastBeepActionTime = currentTime; }
  } else {
    beep(currentBeepTone, currentBeepDuration); beepStep++; lastBeepActionTime = currentTime;
  }
}

void metronomeSVFAT(int ar_time) {
  if (beepSequenceActive) return;
  bool isRunning = (currentState == STATE_SVFAT_COUNTDOWN || currentState == STATE_SVFAT_COUNTUP || currentState == STATE_SVFAT_FACTORIAL_DOWN);
  if (isRunning && ar_time % 300 == 0 && ar_time >= 300 && ar_time != 0) { beep(tone_down, 500); }
  else if (isRunning && ar_time % 150 == 0 && ar_time < 300 && ar_time != 0) { startBeepSequence(3, tone_up, 75); }
  else if (isRunning && ar_time % 100 == 0 && ar_time < 150 && ar_time != 0) { startBeepSequence(2, tone_up, 75); }
  else if (isRunning && ar_time % 10 == 0 && ar_time < 59 && ar_time != 0) { beep(tone_down, 80); }
}

void set_precis() {
  switch (iprec) {
    case 0: precis = 20; break; case 1: precis = 15; break; case 2: precis = 10; break;
    case 3: precis = 7; break; case 4: precis = 5; break; case 5: precis = 3; break;
    case 6: precis = 2; break; case 7: case 8: case 9: case 10: precis = 1; break;
    case 11: precis = 1; break; case 12: precis = 2; break; case 13: precis = 3; break;
    case 14: precis = 4; break; case 15: precis = 6; break; case 16: precis = 8; break;
    case 17: precis = 12; break; case 18: precis = 16; break; case 19: precis = 24; break;
    case 20: precis = 32; break; case 21: precis = 64; break;
  }
  say_precis();
}

void write_eeprom() {
  if (timer_mode == MCDOWN) {
    int start_address = 3 + ((ch - 1) * 20);
    for (int k = 0; k < 10; k++) {
      int current_address = start_address + (k * 2);
      if (k < imcd[ch]) { eeprom_write_word((uint16_t *)current_address, tmcd_time[k][ch]); }
      else { eeprom_write_word((uint16_t *)current_address, 0); }
      delay(5);
    }
  }
  if (timer_mode == SVFAT) {
    int start_address = 89 + ((ch - 1) * 20);
    for (int k = 0; k < 10; k++) {
      int current_address = start_address + (k * 2);
      if (k < isvfat[ch]) { eeprom_write_word((uint16_t *)current_address, tsvfat_time[k][ch]); }
      else { eeprom_write_word((uint16_t *)current_address, 0); }
      delay(5);
    }
  }
}

void load_eeprom() {
  brightness = eeprom_read_word((uint16_t *)1);
  if (brightness == 0xFFFF || brightness > 100) brightness = 80;

  for (byte j = 1; j <= 3; j++) {
    imcd[j] = 0;
    isvfat[j] = 0;
    int mcd_start_address = 3 + ((j - 1) * 20);
    int svfat_start_address = 89 + ((j - 1) * 20);
    for (byte i = 0; i < 10; i++) {
      tmcd_time[i][j] = eeprom_read_word((uint16_t *)(mcd_start_address + i * 2));
      if (tmcd_time[i][j] > 0 && tmcd_time[i][j] != 0xFFFF) { imcd[j] = i + 1; }
      
      tsvfat_time[i][j] = eeprom_read_word((uint16_t *)(svfat_start_address + i * 2));
      if (tsvfat_time[i][j] > 0 && tsvfat_time[i][j] != 0xFFFF) { isvfat[j] = i + 1; }
    }
  }

  for (int index = 1; index <= 3; index++) {
    attrel[index] = eeprom_read_byte((uint8_t *)(83 + index));
    
    // Inizializza i tempi temporanei per canale a 0 (o a un default)
    // Poiché sono int, sono già 0 per default se globali, ma è bene essere sicuri.
    // Per i test, carichiamo un valore base se sono vuoti (es. 10s = 100)
    time_countdown[index] = 0;
    time_dds[index] = 0;
    time_fsttest[index] = 0;
    time_fstdown[index] = 0;
    time_fstdown2[index] = 0;
    time_brn[index] = 0;
    time_test[index] = 0;
  }

  byte preoril_val = eeprom_read_byte((uint8_t *)88);
  if (preoril_val == 0xFF) {
    preoril = LOW; // Default alla pressione
  } else {
    preoril = preoril_val ? HIGH : LOW;
  }
  
  total_relay_seconds = eeprom_read_dword((uint32_t*)eeprom_hour_counter_address);
  if (total_relay_seconds == 0xFFFFFFFF) {
    total_relay_seconds = 0;
  }
}
